(function() {
  "use strict";

  window.LonLat = function(lon, lat) {
    this.lon = lon;
    this.lat = lat;
  };

  window.geofence = {
    MissingLatOrLonException: { message: "each coordinate must have a longitude and latitude" },
    CoordinateParseException: { message: "failed to parse coordinate" },
    NoPolygonsFoundException: { message: "did not find any <Placemark> containing a <Polygon>" },
    MissingZoneTypeException: { message: "could not find the zone type of a polygon" },
    NoInclusionZoneException: { message: "no inclusion zone found" },
    ExcessivePolygonsException: { message: "more than 1500 zones defined" },
    ExcessivePointsException: { message: "more than 6000 points defined" },
    InvalidPolygonException: { message: "a polygon has less than 4 points" },

    SIZE_OF_PACKED_POINT: 8,
    SIZE_OF_PACKED_POLYGON: 8,

    // Returns an array of objects:
    // {
    //   name: string;
    //   coordinates: LatLon[];
    // }
    parseKml: function(xmlSrc) {
      var getNodeText = function(node) {
        var text = "";
        var child;
        for (var i = 0; i < node.childNodes.length; ++i) {
          child = node.childNodes.item(i);
          if (child.nodeType == Node.TEXT_NODE) {
            text = text + child.nodeValue;
          }
        }
        return text;
      };

      // Returns the first child node that is an Element.
      var getFirstChildElement = function(node) {
        var child;
        for(var i = 0; i < node.childNodes.length; ++i) {
          var child = node.childNodes.item(i);
          if(child.nodeType == Node.ELEMENT_NODE)
            return child;
        }
        return null;
      };

      // Takes the text of a <coordinates> element and returns an array of `LonLat` elements.
      var parseCoordinates = function(coordText) {
        var coordStrings = coordText.split(/\s+/);
        // Handle whitespace at the beginning and end of the list of coordinates.
        if (coordStrings.length >= 1 && coordStrings[0].length == 0)
          coordStrings.shift();
        if (coordStrings.length >= 1 && coordStrings[coordStrings.length - 1].length == 0)
          coordStrings.pop();

        var tuple, lon, lat;
        var coords = [];
        for (var i = 0; i < coordStrings.length; ++i) {
          if (coordStrings.length == 0)
            continue;
          tuple = coordStrings[i].split(",");
          if (tuple.length < 2)
            throw this.MissingLatOrLonException;
          lon = parseFloat(tuple[0]);
          lat = parseFloat(tuple[1]);
          if (isNaN(lon) || isNaN(lat))
            throw this.CoordinateParseException;
          coords.push(new LonLat(lon, lat));
        };
        return coords;
      };

      var parseLinearRing = function(linearRingElem, polygon) {
        var child, coordText, coordFound = false;
        var coordStrings, j, tuple, coords;
        for (var i = 0; i < linearRingElem.childNodes.length; ++i) {
          child = linearRingElem.childNodes.item(i);
          if (child.nodeType != Node.ELEMENT_NODE)
            continue;

          if (child.nodeName == "coordinates") {
            coordFound = true;
            coordText = getNodeText(child);
            polygon.coordinates = parseCoordinates(coordText);
          }
        }

        if (!coordFound)
          throw { message: "<LinearRing> must contain <coordinates> element" };
      };

      var parsePolygon = function(polygonElem, polygon) {
        var child, firstChild, outerBoundFound = false;
        for (var i = 0; i < polygonElem.childNodes.length; ++i) {
          child = polygonElem.childNodes.item(i);
          if (child.nodeType != Node.ELEMENT_NODE)
            continue;

          if (child.nodeName == "outerBoundaryIs") {
            outerBoundFound = true;
            firstChild = getFirstChildElement(child);
            if (!firstChild || firstChild.nodeName != "LinearRing")
              throw { message: "<outerBoundaryIs> must contain <LinearRing> element" };

            parseLinearRing(firstChild, polygon);
          } else if (child.nodeName == "innerBoundaryIs") {
            throw { message: "<innerBoundaryIs> not currently supported" };
          }
        }

        if (!outerBoundFound)
          throw { message: "<Polygon> must contain <outerBoundaryIs> element" };
      };

      // Returns an object:
      // {
      //   name: string;
      //   coordinates: LatLon[];
      // }
      var parsePlacemark = function(placemarkElem) {
        var polygon = {};
        var child;
        for (var i = 0; i < placemarkElem.childNodes.length; ++i) {
          child = placemarkElem.childNodes.item(i);
          if (child.nodeType != Node.ELEMENT_NODE)
            continue;

          if (child.nodeName == "name") {
            polygon.name = getNodeText(child);
          } else if (child.nodeName == "Polygon") {
            parsePolygon(child, polygon);
          }
        }
        return polygon;
      };


	  var parseChildren = function(rootElem) {
		var child, polygon, i;
		var polygons = [];
		
		for (i = 0; i < rootElem.childNodes.length; ++i) {
			child = rootElem.childNodes.item(i);
			if (child.nodeType != Node.ELEMENT_NODE)
			  continue;

			if (child.nodeName == "Placemark") {
			  polygon = parsePlacemark(child);
			  if (polygon)
				polygons.push(polygon);
			}
			else if (child.nodeName == "Folder" || child.nodeName == "Document") {
				var childPolys = parseChildren(child);
				
				if (childPolys)
					polygons.push.apply(polygons, childPolys);
			}
		}
		
		return polygons;
	  }

      var doc, kmlElem, rootElem;
      var polygons = [];
      
      doc = new DOMParser().parseFromString(xmlSrc, "text/xml");
	  
	  // Test if XML file is actually KML format
      kmlElem = getFirstChildElement(doc);
      if (!kmlElem || kmlElem.nodeType != Node.ELEMENT_NODE || kmlElem.nodeName != "kml")
        throw { message: "document must contain <kml> element" };
	
      rootElem = getFirstChildElement(kmlElem);
      if (!rootElem || rootElem.nodeType != Node.ELEMENT_NODE)
        throw { message: "<kml> must contain a root element" };
	
	  if (rootElem.nodeName != "Document" && rootElem.nodeName != "Folder")
		throw { message: "<kml> must contain <Document> or <Folder> as root element" };

	  polygons = parseChildren(rootElem);
      
      if (polygons.length == 0)
        throw this.NoPolygonsFoundException;

      return polygons;
    },

	sortPolygons: function(polygons) {
		
	  // Sort polygons by area
	  // Assumes no intersecting polygons, self-intersecting polygons,
	  // or other similar conditions
		
	  var getPolygonArea = function (polygon) {
      // Assumes vertecies are in order, and polygon isn't self intersecting
      var polyArea = 0;
      if (typeof(polygon.coordinates) == 'undefined')
          throw { message: "a polygon has no points" };

      var numVertex = polygon.coordinates.length;
      var i = 0;
      var j = numVertex - 1;
      for (; i < numVertex; j = i++) {
        var pt0 = polygon.coordinates[i];
        var pt1 = polygon.coordinates[j];
        
        polyArea += (pt1.lon + pt0.lon) * (pt1.lat - pt0.lat);
      }
      
      return Math.abs(polyArea / 2);
    };
      
    var polygonCompareFunction = function(a,b) {
       
      var areaA = getPolygonArea(a);
      var areaB = getPolygonArea(b);
      
      return areaB - areaA;
	  };
	  
	  // Sort polygons by area
	  polygons.sort(polygonCompareFunction);
	  
	  return polygons;
	},
	
    _setInt16: function(arr, index, n) {
      arr[index    ] =  n        & 0xFF;
      arr[index + 1] = (n >>  8) & 0xFF;
    },

    _setInt32: function(arr, index, n) {
      arr[index    ] =  n        & 0xFF;
      arr[index + 1] = (n >>  8) & 0xFF;
      arr[index + 2] = (n >> 16) & 0xFF;
      arr[index + 3] = (n >> 24) & 0xFF;
    },

    // Returns an object:
    // {
    //   points: Uint8Array;
    //   polygons: Uint8Array;
    // }
    packPolygonData: function(polygons) {
      var polygon, polygonStart, point, lon, lat;
      var pointCount = polygons.reduce(function(sum, p) { return sum + p.coordinates.length; }, 0);
      
      var pointData = new Uint8Array(this.SIZE_OF_PACKED_POINT * pointCount);
      var pointIndex = 0;
      var polygonData = new Uint8Array(this.SIZE_OF_PACKED_POLYGON * polygons.length);
      var polygonIndex = 0;
      for (var i = 0; i < polygons.length; ++i) {
        polygon = polygons[i];
        polygonStart = polygonIndex * this.SIZE_OF_PACKED_POLYGON;
		
		// startPtIndex
        this._setInt16(polygonData, polygonStart, pointIndex);
        // ptCount
        this._setInt16(polygonData, polygonStart + 2, polygon.coordinates.length);
		
        // isInclusionZone
        if (/inclusion/i.test(polygon.name))
          polygonData[polygonStart + 4] = 1;
        else if (/exclusion/i.test(polygon.name))
          polygonData[polygonStart + 4] = 0;
        else
          throw this.MissingZoneTypeException;
        
		// rfu
        polygonData[polygonStart + 5] = 0;
		polygonData[polygonStart + 6] = 0;
		polygonData[polygonStart + 7] = 0;

        for (var j = 0; j < polygon.coordinates.length; ++j) {
          point = polygon.coordinates[j];
          lon = Math.round(point.lon * 1e7);
          lat = Math.round(point.lat * 1e7);
          this._setInt32(pointData, pointIndex * this.SIZE_OF_PACKED_POINT,     lat);
          this._setInt32(pointData, pointIndex * this.SIZE_OF_PACKED_POINT + 4, lon);
          ++pointIndex;
        }
        ++polygonIndex;
      }

      return { points: pointData, polygons: polygonData };
    },

    loadPolygonDataFromKml: function(xmlSrc) {
      var polygons = this.parseKml(xmlSrc);
	  
      // Sort polygons assuming that the highest priority (highest index)
      // will be the smallest polygon
      polygons = this.sortPolygons(polygons);
    
      // Ensure there is an inclusion zone (per FAA-E-3032)
      var foundInclusionZone = false;
      var totalPoints = 0;
      for (var i = 0; i < polygons.length; ++i) {		
        // isInclusionZone
        if (/inclusion/i.test(polygons[i].name))
          foundInclusionZone = true;
        if (typeof(polygons[i].coordinates) == 'undefined' || polygons[i].coordinates.length < 4)
          throw this.InvalidPolygonException;

        totalPoints += polygons[i].coordinates.length;
      }        
      
      if (!foundInclusionZone)
        throw this.NoInclusionZoneException;
      
      if (polygons.length > 1500)
        throw this.ExcessivePolygonsException;
      
      if (totalPoints > 6000)
        throw this.ExcessivePointsException;

      return this.packPolygonData(polygons);
    }
  };
})();
